import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:shared_preferences/shared_preferences.dart';

import 'package:fl_sevengen_society_user_app/localization/localization_const.dart';
import 'package:fl_sevengen_society_user_app/theme/theme.dart';

class SettingsScreen extends StatefulWidget {
  const SettingsScreen({super.key});

  @override
  State<SettingsScreen> createState() => _SettingsScreenState();
}

class _SettingsScreenState extends State<SettingsScreen> {
  bool _isLoggedIn = false;
  bool _loading = true;

  @override
  void initState() {
    super.initState();
    _checkLoginStatus();
  }

  Future<void> _checkLoginStatus() async {
    try {
      final prefs = await SharedPreferences.getInstance();
      final ridNo = prefs.getString('rid_no');
      if (!mounted) return;
      setState(() {
        _isLoggedIn = ridNo != null && ridNo.isNotEmpty;
        _loading = false;
      });
    } catch (e) {
      if (!mounted) return;
      setState(() {
        _isLoggedIn = false;
        _loading = false;
      });
    }
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        backgroundColor: primaryColor,
        elevation: 0.0,
        titleSpacing: 0.0,
        centerTitle: false,
        automaticallyImplyLeading: false,
        leading: IconButton(
          onPressed: () {
            // Navigate to BottomBar
            Navigator.pushReplacementNamed(context, '/bottomBar');
          },
          icon: const Icon(
            Icons.arrow_back,
            color: Colors.white,
          ),
        ),
        title: Text(
          getTranslate(context, 'settings.settings'),
          style: TextStyle(
            fontWeight: FontWeight.w600,
            fontSize: 18,
            color: Colors.white,
          ),
        ),
      ),
      body: _loading
          ? const Center(child: CircularProgressIndicator())
          : ListView(
        padding: const EdgeInsets.symmetric(horizontal: fixPadding * 2.0),
        physics: const BouncingScrollPhysics(),
        children: [
          // About RMBF
          optionWidget(
            Icons.info_outline,
            getTranslate(context, 'settings.about_rmbf'),
                () {
              Navigator.pushNamed(context, '/AboutRmbf');
            },
          ),
          // Edit Profile
          optionWidget(
            Icons.person_outline,
            getTranslate(context, 'settings.edit_profile'),
                () {
              Navigator.pushNamed(context, '/editProfile');
            },
          ),
          // Language
          optionWidget(
            CupertinoIcons.globe,
            getTranslate(context, 'settings.language'),
                () {
              Navigator.pushNamed(context, '/language');
            },
          ),
          // Privacy Policy
          optionWidget(
            Icons.privacy_tip_outlined,
            getTranslate(context, 'settings.privacy_policy'),
                () {
              Navigator.pushNamed(context, '/privacyPolicy');
            },
          ),
          // Delete Account
          optionWidget(
            Icons.delete_forever_outlined,
            getTranslate(context, 'settings.delete_account'),
                () {
              Navigator.pushNamed(context, '/deleteAccount');
            },
            color: lightRedColor,
            iconColor: lightRedColor,
          ),
          // Login/Logout
          if (_isLoggedIn)
            optionWidget(
              Icons.logout,
              getTranslate(context, 'settings.logout'),
                  () {
                logoutDialog(context);
              },
              color: lightRedColor,
              iconColor: lightRedColor,
            )
          else
            optionWidget(
              Icons.login,
              getTranslate(context, 'settings.login'),
                  () {
                Navigator.pushNamed(context, '/login');
              },
              color: primaryColor,
              iconColor: primaryColor,
            ),
        ],
      ),
    );
  }

  /// Logout confirmation dialog
  logoutDialog(BuildContext context) {
    return showDialog<void>(
      context: context,
      builder: (context) {
        return Dialog(
          clipBehavior: Clip.hardEdge,
          shape: RoundedRectangleBorder(
            borderRadius: BorderRadius.circular(10.0),
          ),
          insetPadding: const EdgeInsets.symmetric(
            horizontal: fixPadding * 4.5,
            vertical: fixPadding * 2.0,
          ),
          backgroundColor: whiteColor,
          child: ListView(
            shrinkWrap: true,
            physics: const BouncingScrollPhysics(),
            children: [
              Padding(
                padding: const EdgeInsets.all(fixPadding * 2.0),
                child: Column(
                  children: [
                    const Icon(
                      Icons.logout,
                      color: primaryColor,
                      size: 35,
                    ),
                    heightSpace,
                    Text(
                      getTranslate(context, 'settings.logout'),
                      style: semibold18Primary,
                    ),
                    heightSpace,
                    Text(
                      getTranslate(context, 'settings.logout_text'),
                      style: semibold16Black33,
                      textAlign: TextAlign.center,
                    )
                  ],
                ),
              ),
              Row(
                children: [
                  Expanded(
                    child: GestureDetector(
                      onTap: () {
                        Navigator.pop(context);
                      },
                      child: Container(
                        padding: const EdgeInsets.symmetric(
                          vertical: fixPadding * 1.2,
                        ),
                        decoration: BoxDecoration(
                          color: whiteColor,
                          boxShadow: [
                            BoxShadow(
                              color: blackColor.withOpacity(0.2),
                              blurRadius: 6.0,
                            )
                          ],
                        ),
                        alignment: Alignment.center,
                        child: Text(
                          getTranslate(context, 'settings.no'),
                          style: semibold18Black33,
                        ),
                      ),
                    ),
                  ),
                  Expanded(
                    child: GestureDetector(
                      onTap: () async {
                        try {
                          final prefs = await SharedPreferences.getInstance();
                          await prefs.clear();
                        } catch (e) {
                          // ignore error
                        }

                        if (!mounted) return;
                        Navigator.pushNamedAndRemoveUntil(
                          context,
                          '/login',
                              (route) => false,
                        );
                      },
                      child: Container(
                        padding: const EdgeInsets.symmetric(
                          vertical: fixPadding * 1.2,
                        ),
                        decoration: BoxDecoration(
                          color: primaryColor,
                          boxShadow: [
                            BoxShadow(
                              color: blackColor.withOpacity(0.2),
                              blurRadius: 6.0,
                            )
                          ],
                        ),
                        alignment: Alignment.center,
                        child: Text(
                          getTranslate(context, 'settings.yes'),
                          style: semibold18White,
                        ),
                      ),
                    ),
                  ),
                ],
              )
            ],
          ),
        );
      },
    );
  }

  Widget optionWidget(
      IconData icon,
      String title,
      Function() onTap, {
        Color color = black33Color,
        Color iconColor = primaryColor,
      }) {
    return GestureDetector(
      onTap: onTap,
      child: Container(
        padding: const EdgeInsets.symmetric(
          vertical: fixPadding * 0.8,
          horizontal: fixPadding,
        ),
        margin: const EdgeInsets.symmetric(vertical: fixPadding),
        decoration: BoxDecoration(
          color: whiteColor,
          borderRadius: BorderRadius.circular(5.0),
          boxShadow: [
            BoxShadow(
              color: shadowColor.withOpacity(0.2),
              blurRadius: 6.0,
            ),
          ],
        ),
        child: Row(
          children: [
            Container(
              height: 35,
              width: 35,
              decoration: BoxDecoration(
                color: f6f3Color,
                borderRadius: BorderRadius.circular(5.0),
              ),
              alignment: Alignment.center,
              child: Icon(
                icon,
                color: iconColor,
              ),
            ),
            widthSpace,
            Expanded(
              child: Text(
                title,
                style: medium15Black33.copyWith(color: color),
              ),
            ),
            widthSpace,
            const Icon(
              Icons.arrow_forward_ios,
              color: black33Color,
              size: 18.0,
            )
          ],
        ),
      ),
    );
  }
}
